/*
 * Definitions for communicating with a Fabric Management Server
 */
#ifndef _fms_comm_
#define _fms_comm_

#define FMS_SERVICE_PORT 3333	/* default port for FMS service socket */

/* version numbers */
#define LF_VERSION 9

/*
 * message which defines a connection type.  This is the first thing sent
 * across an IP connection once one is established.
 */


/* connection types */
typedef enum {
  FMS_CONN_FMA='a',	/* an individual fabric management agent */
  FMS_CONN_GENERIC='g',	/* an application wishes to make a request */
  FMS_CONN_PEER='p'	/* a peer fabric management server */
} fms_conn_type_t;


/*
 * Common header
 */
struct fms_msg_header {	/* header */
  uint32_t length_32;
  uint32_t msg_type_32;
};

/*
 * FMA->FMS messages
 */

enum {		/* FMA->FMS message types */
  FMA_FMS_IDENT_MSG,
  FMA_FMS_HOST_MSG,
  FMA_FMS_MAP_IS_INVALID,
  FMA_FMS_MAP_UPLOAD,
  FMA_FMS_SEND_INVALID_ROUTE_COMPLETE,
  FMA_FMS_SRAM_PARITY_ERROR,
  FMA_FMS_NIC_ERROR,
  FMA_FMS_NIC_BADCRC,
  FMA_FMS_PROXY_FMA_TO_FMS,
  FMA_FMS_NUM_MSGS
};

union fma_fms_msg {

  /* self-identification message from FMA */
  struct fma_fms_ident_msg {
    uint32_t fma_version_32;
    char hostname[LF_STRING_LEN];
  } ident;

  /* host description message */
  struct fma_fms_host_msg {

    uint32_t sw_type_32;
    uint32_t sw_version_32;
    uint32_t nic_cnt_32;		/* how many NICs */

    struct fma_fms_nic {
      char product_id[LF_SHORT_STRING_LEN];	/* type of NIC */
      char serial_no[LF_SHORT_STRING_LEN];	/* serial number */

      /* xbar ID and port to which each interface is connected, 
       * xbar_id is -1 if apparently disconnected */
      int32_t xbar_id_32[LF_MAX_NIC_PORTS];
      uint32_t xbar_port_32[LF_MAX_NIC_PORTS];

      uint32_t num_active_ports_32;	/* number of SW-supported NIC ports */

      lf_mac_addr_t mac_addr;		/* MAC address */
      uint16_t nic_id_16;		/* host nic ID */
    } nic_array[1];
  } host;

  /* FMA's current map is invalid */
  struct fma_fms_map_invalid_msg {
    uint32_t minv_map_version_32;
    uint8_t minv_use_me_8;	/* TRUE if this fma is good choice for mapper */
    char why[LF_STRING_LEN];
  } map_invalid;

  /* FMA reports NIC has an error */
  struct fma_fms_nic_error_msg {
    uint32_t nic_id_32;		/* NIC index */
    uint32_t error_32;		/* the error code */
  } nic_error;

  /* FMA reports NIC has too many badcrcs */
  struct fma_fms_nic_badcrc_msg {
    uint32_t nic_id_32;		/* NIC index */
    uint32_t port_32;		/* NIC port */
    uint32_t badcrcs_32;	/* badcrcs during interval */
  } nic_badcrc;

#define FMA_FMS_HOST_MSG_SIZE(NICS) \
   (sizeof(struct fma_fms_host_msg) + (((NICS)-1) * sizeof(struct fma_fms_nic)))

  /* Results of FMA send to invalid routes from FMA */
  struct fma_fms_inv_rt_msg {
    uint32_t irr_seq_no_32;
    uint32_t irr_send_succeeded_32;
    uint32_t irr_xbar_index_32;
    uint32_t irr_host_in_port_32;
  } inv_rt;

  /* Message from an FMA destined for a the FMS */
  struct lf_proxy_fma_to_fms {
    struct lf_proxy_fma_to_fms_hdr {
      uint32_t proxy_client_id_32;
      char hostname[LF_STRING_LEN];

      uint8_t return_route[MYRI_MAX_ROUTE_LEN];
      uint8_t return_route_len_8;

      uint8_t nic_index_8;
      uint8_t port_8;
      uint8_t pad[5];

      uint32_t msg_type_32;	/* type and length of embedded message */
      uint32_t length_32;
    } h;

    uint8_t data[1];		/* start of data */
  } proxy_fma_to_fms;
};


/*
 * Client <-> FMS
 */

enum {		/* generic->FMS message types */
  LF_CLIENT_FMS_NOTIFY_REQUEST,
  LF_CLIENT_FMS_NOTIFY_CANCEL,
  LF_CLIENT_FMS_LIST_NOTIFY_REGISTRY,
  LF_CLIENT_FMS_QUERY_ALERTS,
  LF_CLIENT_FMS_ACK_ALERT,
  LF_CLIENT_FMS_QUERY_STATUS,
  LF_CLIENT_FMS_SWITCH_MAINTENANCE,
  LF_CLIENT_FMS_QUERY_PROXIMITY,
  LF_CLIENT_FMS_NUM_MSGS
};

/*
 * Definitions for registering for event notification
 */
enum {
  FMS_NOTIFY_EMAIL,		/* send email */
  FMS_NOTIFY_EXEC,		/* exec a program */
  FMS_NOTIFY_FILE,		/* append to a file */
  FMS_NOTIFY_TCP		/* send alerts over this socket */
};

enum {
  FMS_EVENT_INFO=1,		/* no operational impact */
  FMS_EVENT_ERROR=2,		/* minimal operational impact */
  FMS_EVENT_CRITICAL=4,		/* needs attention */
  FMS_EVENT_ALERT=8,		/* an alert */
  FMS_EVENT_DEBUG=16,		/* developer only */
  FMS_EVENT_ALL=0xFFFF		/* mask matching all */
};

/*
 * Union of all client->FMS messages
 */
union lf_client_fms_msg {

  /* request notifitcation of events */
  struct fms_notify_request {
    uint8_t notify_type;
    uint8_t persistant;			/* saved in database? */
    uint16_t event_types_16;		/* bitfield of event types */
    union {
      char email[LF_STRING_LEN];
      char program[LF_STRING_LEN];	/* for FMS_NOTIFY_EXEC */
      char filename[LF_STRING_LEN];	/* for FMS_NOTIFY_FILE */
    } u;
  } notify_request;

  /* query outstanding alerts */
  struct fms_query_alerts {
    uint8_t show_acked;		/* include ACKed alerts */
    uint8_t show_relics;	/* include relic alerts */
  } query_alerts;

  /* ACK an alert */
  struct fms_ack_alert {
    uint32_t alert_id_32;		/* alert to ACK */
  } ack_alert;

  /* FMS status query */
  struct fms_query_status {
    uint32_t flags_32;
  } query_status;

  /* FMS switch maintenance */
  struct fms_switch_maintenance {
    lf_string_t enc_name;
    int8_t slot;		/* -1 means all slots */
    int8_t port_label;		/* -1 means all ports */
    uint8_t up;
  } switch_maintenance;
};

/*
 * FMS replies to messages
 */

/* FMS status reply */
struct lf_fms_status_msg {
  uint32_t hosts_known_32;
  uint32_t fmas_found_32;
  uint32_t unacked_alerts_32;
  uint8_t mapping_complete_8;
  lf_string_t last_mapper;
  uint8_t db_complete_8;
};

/* generic return code message */
struct lf_fms_return_code {
  int32_t fms_rc_32;
  lf_string_t reason;
};


#endif	/* _fms_comm_ */
